<?php
require_once 'includes/db.php';

// Helper escape
function e($str){ return htmlspecialchars((string)$str, ENT_QUOTES, 'UTF-8'); }

// Validasi ID
if (!isset($_GET['id']) || !is_numeric($_GET['id'])) {
    header("Location: index.php");
    exit();
}
$product_id = (int) $_GET['id'];

// Ambil produk
$stmt = $conn->prepare("
    SELECT p.id, p.nama_produk, p.deskripsi, p.harga, p.gambar, p.kategori_id, k.nama_kategori
    FROM produk p
    LEFT JOIN kategori k ON k.id = p.kategori_id
    WHERE p.id = ?
    LIMIT 1
");
$stmt->bind_param("i", $product_id);
$stmt->execute();
$res = $stmt->get_result();

$produk = $res->num_rows ? $res->fetch_assoc() : null;
$page_title = $produk ? $produk['nama_produk'] : 'Produk Tidak Ditemukan';

// Siapkan produk terkait jika ada produk
$related = [];
if ($produk && !empty($produk['kategori_id'])) {
    $stmt2 = $conn->prepare("
        SELECT id, nama_produk, harga, gambar
        FROM produk
        WHERE kategori_id = ? AND id <> ?
        ORDER BY id DESC
        LIMIT 8
    ");
    $stmt2->bind_param("ii", $produk['kategori_id'], $product_id);
    $stmt2->execute();
    $related = $stmt2->get_result()->fetch_all(MYSQLI_ASSOC);
    $stmt2->close();
}

$stmt->close();
$conn->close();
?>
<!DOCTYPE html>
<html lang="id">
<head>
  <meta charset="UTF-8" />
  <meta name="viewport" content="width=device-width, initial-scale=1.0"/>
  <title><?= e($page_title); ?> - Kedai Salsabilla</title>
  <meta name="description" content="<?= $produk ? e(mb_substr(strip_tags($produk['deskripsi'] ?? ''),0,150)) : 'Produk Kedai Salsabilla'; ?>">
  <link rel="preconnect" href="https://fonts.googleapis.com">
  <link rel="preconnect" href="https://fonts.gstatic.com" crossorigin>
  <link href="https://fonts.googleapis.com/css2?family=Inter:wght@400;500;600;700;800&display=swap" rel="stylesheet">
  <link rel="icon" href="data:,">
  <style>
    :root{
      --primary: #ff6b35;
      --primary-dark: #e55a2e;
      --primary-light: #ff8559;
      --secondary: #f39c12;
      --accent: #3498db;
      --success: #27ae60;
      --dark: #2c3e50;
      --gray-50: #f8fafc;
      --gray-100: #f1f5f9;
      --gray-200: #e2e8f0;
      --gray-300: #cbd5e1;
      --gray-400: #94a3b8;
      --gray-500: #64748b;
      --gray-600: #475569;
      --gray-700: #334155;
      --gray-800: #1e293b;
      --gray-900: #0f172a;
      --white: #ffffff;
      --shadow-sm: 0 1px 2px rgba(0,0,0,0.05);
      --shadow: 0 4px 15px rgba(0,0,0,0.08);
      --shadow-lg: 0 10px 40px rgba(0,0,0,0.12);
      --shadow-xl: 0 25px 50px rgba(0,0,0,0.15);
      --radius: 16px;
      --radius-lg: 20px;
      --gradient-bg: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
      --gradient-card: linear-gradient(145deg, #ffffff 0%, #f8fafc 100%);
    }

    * {
      box-sizing: border-box;
      margin: 0;
      padding: 0;
    }

    html {
      scroll-behavior: smooth;
    }

    body {
      font-family: 'Inter', -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, sans-serif;
      background: linear-gradient(135deg, #f5f7fa 0%, #c3cfe2 100%);
      color: var(--gray-800);
      line-height: 1.6;
      min-height: 100vh;
    }

    /* Remove all underlines */
    a {
      text-decoration: none;
      color: inherit;
      transition: all 0.3s ease;
    }

    .container {
      max-width: 1200px;
      margin: 0 auto;
      padding: 0 24px;
    }

    /* Enhanced Header */
    .site-header {
      position: sticky;
      top: 0;
      z-index: 1000;
      background: rgba(255, 255, 255, 0.95);
      backdrop-filter: blur(20px);
      border-bottom: 1px solid var(--gray-200);
      box-shadow: var(--shadow-sm);
    }

    .nav {
      display: flex;
      align-items: center;
      justify-content: space-between;
      padding: 16px 0;
    }

    .brand {
      display: flex;
      gap: 12px;
      align-items: center;
      font-weight: 700;
      color: var(--primary);
      font-size: 18px;
    }

    .brand .logo {
      width: 44px;
      height: 44px;
      border-radius: 14px;
      background: linear-gradient(135deg, var(--primary) 0%, var(--primary-light) 100%);
      display: grid;
      place-items: center;
      color: white;
      font-weight: 800;
      font-size: 16px;
      box-shadow: var(--shadow);
    }

    .nav-cta {
      display: flex;
      gap: 8px;
      align-items: center;
    }

    .nav-cta a {
      padding: 10px 20px;
      border-radius: 10px;
      font-weight: 600;
      font-size: 14px;
      transition: all 0.3s ease;
      border: 2px solid transparent;
    }

    .nav-cta a:first-child {
      color: var(--gray-600);
      background: var(--gray-100);
    }

    .nav-cta a:first-child:hover {
      background: var(--gray-200);
      color: var(--gray-700);
    }

    .nav-cta a.primary {
      background: linear-gradient(135deg, var(--primary) 0%, var(--primary-light) 100%);
      color: white;
      box-shadow: 0 4px 15px rgba(255, 107, 53, 0.3);
    }

    .nav-cta a.primary:hover {
      transform: translateY(-2px);
      box-shadow: 0 8px 25px rgba(255, 107, 53, 0.4);
    }

    /* Enhanced Page Head */
    .page-head {
      padding: 32px 0 20px;
    }

    .breadcrumb {
      font-size: 14px;
      color: var(--gray-500);
      margin-bottom: 12px;
      display: flex;
      align-items: center;
      gap: 8px;
    }

    .breadcrumb a {
      color: var(--primary);
      font-weight: 500;
      padding: 4px 8px;
      border-radius: 6px;
      transition: all 0.2s ease;
    }

    .breadcrumb a:hover {
      background: var(--gray-100);
      color: var(--primary-dark);
    }

    .breadcrumb::before {
      content: "🏠";
      margin-right: 4px;
    }

    .title {
      margin: 0;
      font-size: clamp(28px, 5vw, 42px);
      font-weight: 800;
      color: var(--gray-900);
      line-height: 1.2;
      background: linear-gradient(135deg, var(--primary) 0%, var(--secondary) 100%);
      -webkit-background-clip: text;
      -webkit-text-fill-color: transparent;
      background-clip: text;
    }

    /* Enhanced Detail Card */
    .detail {
      margin: 24px 0;
      display: grid;
      gap: 32px;
      grid-template-columns: 1fr;
      background: var(--gradient-card);
      border: 1px solid var(--gray-200);
      border-radius: var(--radius-lg);
      box-shadow: var(--shadow-lg);
      padding: 32px;
      position: relative;
      overflow: hidden;
    }

    .detail::before {
      content: '';
      position: absolute;
      top: 0;
      left: 0;
      right: 0;
      height: 4px;
      background: linear-gradient(90deg, var(--primary) 0%, var(--secondary) 100%);
    }

    @media (min-width: 900px) {
      .detail {
        grid-template-columns: 1.3fr 1fr;
        align-items: start;
      }
    }

    /* Enhanced Media */
    .media {
      position: relative;
      border-radius: var(--radius);
      overflow: hidden;
      border: 1px solid var(--gray-200);
      background: var(--gray-100);
      aspect-ratio: 4/3;
      box-shadow: var(--shadow);
    }

    .media img {
      width: 100%;
      height: 100%;
      object-fit: cover;
      display: block;
      transition: transform 0.3s ease;
    }

    .media:hover img {
      transform: scale(1.05);
    }

    /* Enhanced Info Section */
    .info h2 {
      display: none;
    }

    .price {
      font-size: 32px;
      color: var(--primary);
      font-weight: 900;
      margin: 8px 0 16px;
      display: flex;
      align-items: center;
      gap: 8px;
    }

    .price::before {
      content: "💰";
      font-size: 24px;
    }

    .desc {
      color: var(--gray-600);
      font-size: 16px;
      line-height: 1.7;
      background: var(--gray-50);
      padding: 20px;
      border-radius: 12px;
      border-left: 4px solid var(--primary);
      margin: 16px 0;
    }

    /* Enhanced Buttons */
    .cta-row {
      margin-top: 24px;
      display: flex;
      gap: 12px;
      flex-wrap: wrap;
    }

    .btn {
      appearance: none;
      border: none;
      border-radius: 12px;
      padding: 14px 24px;
      font-weight: 700;
      cursor: pointer;
      font-size: 15px;
      transition: all 0.3s ease;
      position: relative;
      overflow: hidden;
      display: inline-flex;
      align-items: center;
      gap: 8px;
    }

    .btn::before {
      content: '';
      position: absolute;
      top: 0;
      left: -100%;
      width: 100%;
      height: 100%;
      background: linear-gradient(90deg, transparent, rgba(255,255,255,0.2), transparent);
      transition: left 0.5s;
    }

    .btn:hover::before {
      left: 100%;
    }

    .btn.wa {
      background: linear-gradient(135deg, var(--success) 0%, #2ecc71 100%);
      color: white;
      box-shadow: 0 6px 20px rgba(46, 204, 113, 0.3);
    }

    .btn.wa:hover {
      transform: translateY(-3px);
      box-shadow: 0 12px 30px rgba(46, 204, 113, 0.4);
    }

    .btn.ghost {
      background: var(--white);
      color: var(--gray-700);
      border: 2px solid var(--gray-300);
      box-shadow: var(--shadow-sm);
    }

    .btn.ghost:hover {
      background: var(--gray-50);
      border-color: var(--primary);
      color: var(--primary);
      transform: translateY(-2px);
    }

    /* Enhanced Section Title */
    .section-title {
      display: flex;
      align-items: center;
      gap: 16px;
      margin: 40px 0 24px;
      position: relative;
    }

    .section-title .line {
      height: 2px;
      background: linear-gradient(90deg, transparent, var(--primary), transparent);
      flex: 1;
    }

    .section-title h3 {
      margin: 0;
      font-size: 20px;
      font-weight: 700;
      color: var(--gray-800);
      background: white;
      padding: 8px 20px;
      border-radius: 20px;
      box-shadow: var(--shadow);
      white-space: nowrap;
    }

    /* Enhanced Product Grid */
    .product-grid {
      display: grid;
      gap: 20px;
      grid-template-columns: repeat(auto-fill, minmax(240px, 1fr));
    }

    .product-card-link {
      display: block;
      transition: transform 0.3s ease;
    }

    .product-card-link:hover {
      transform: translateY(-8px);
    }

    .card {
      background: var(--white);
      border: 1px solid var(--gray-200);
      border-radius: var(--radius);
      box-shadow: var(--shadow);
      overflow: hidden;
      transition: all 0.3s ease;
      height: 100%;
      position: relative;
    }

    .card::before {
      content: '';
      position: absolute;
      top: 0;
      left: 0;
      right: 0;
      height: 3px;
      background: linear-gradient(90deg, var(--primary) 0%, var(--secondary) 100%);
      transform: scaleX(0);
      transition: transform 0.3s ease;
    }

    .card:hover::before {
      transform: scaleX(1);
    }

    .card:hover {
      box-shadow: var(--shadow-xl);
      border-color: var(--primary);
    }

    .card .media-sm {
      height: 160px;
      background: var(--gray-100);
      overflow: hidden;
      position: relative;
    }

    .card .media-sm img {
      width: 100%;
      height: 100%;
      object-fit: cover;
      transition: transform 0.3s ease;
    }

    .card:hover .media-sm img {
      transform: scale(1.1);
    }

    .card .body {
      padding: 16px;
      text-align: center;
    }

    .card .body h4 {
      margin: 0 0 8px 0;
      font-size: 16px;
      font-weight: 600;
      color: var(--gray-800);
      line-height: 1.3;
    }

    .card .price {
      color: var(--primary);
      font-weight: 800;
      font-size: 18px;
      margin: 0;
    }

    .card .price::before {
      display: none;
    }

    /* 404 Styles */
    .not-found {
      text-align: center;
      padding: 60px 20px;
    }

    .not-found::before {
      content: "😔";
      font-size: 64px;
      display: block;
      margin-bottom: 16px;
    }

    /* Enhanced Footer */
    footer {
      margin: 60px 0 24px;
      padding: 24px 0;
      border-top: 1px solid var(--gray-200);
      text-align: center;
      color: var(--gray-500);
      background: var(--gray-50);
      border-radius: var(--radius) var(--radius) 0 0;
    }

    /* Responsive Design */
    @media (max-width: 640px) {
      .container {
        padding: 0 16px;
      }
      
      .detail {
        padding: 20px;
        gap: 20px;
      }
      
      .cta-row {
        flex-direction: column;
      }
      
      .btn {
        width: 100%;
        justify-content: center;
      }
      
      .nav-cta {
        flex-direction: column;
        gap: 8px;
      }
      
      .nav-cta a {
        padding: 8px 16px;
        font-size: 13px;
      }
    }

    /* Loading Animation */
    @keyframes shimmer {
      0% { background-position: -200px 0; }
      100% { background-position: calc(200px + 100%) 0; }
    }

    .loading {
      background: linear-gradient(90deg, #f0f0f0 25%, #e0e0e0 50%, #f0f0f0 75%);
      background-size: 200px 100%;
      animation: shimmer 1.5s infinite;
    }

    /* Accessibility */
    @media (prefers-reduced-motion: reduce) {
      * {
        animation-duration: 0.01ms !important;
        animation-iteration-count: 1 !important;
        transition-duration: 0.01ms !important;
      }
    }

    .sr-only {
      position: absolute;
      width: 1px;
      height: 1px;
      padding: 0;
      margin: -1px;
      overflow: hidden;
      clip: rect(0, 0, 0, 0);
      white-space: nowrap;
      border: 0;
    }
  </style>
</head>
<body>
  <header class="site-header" role="banner">
    <div class="container nav" aria-label="Navigasi utama">
      <div class="brand">
        <div class="logo">KS</div>
        <div>Kedai Salsabilla</div>
      </div>
      <div class="nav-cta">
        <a href="index.php">Beranda</a>
        <a class="primary" href="https://wa.me/6283821470781?text=Halo,%20saya%20mau%20pesan%20menu%20dari%20Kedai%20%20Salsabilla" target="_blank" rel="noopener">Pesan via WA</a>
      </div>
    </div>
  </header>

  <main class="container" role="main">
    <?php if(!$produk): ?>
      <section class="page-head">
        <p class="breadcrumb"><a href="index.php">Home</a> / 404</p>
        <h1 class="title">Produk Tidak Ditemukan</h1>
      </section>
      <div class="detail not-found">
        <p>Maaf, produk yang Anda cari tidak ada atau sudah dihapus.</p>
        <a class="btn ghost" href="index.php">Kembali ke Beranda</a>
      </div>
    <?php else: ?>
      <!-- Header section -->
      <section class="page-head" aria-labelledby="judul-produk">
        <p class="breadcrumb">
          <a href="index.php">Home</a>
          <?php if(!empty($produk['nama_kategori'])): ?>
            / <a href="index.php?kategori_id=<?= (int)$produk['kategori_id'];?>#menu"><?= e($produk['nama_kategori']);?></a>
          <?php endif; ?>
          / Detail
        </p>
        <h1 id="judul-produk" class="title"><?= e($produk['nama_produk']); ?></h1>
      </section>

      <!-- Detail Card -->
      <section class="detail" aria-label="Detail produk">
        <div class="media">
          <img
            src="<?= !empty($produk['gambar'])
                ? 'uploads/'.e($produk['gambar'])
                : 'data:image/svg+xml,<svg xmlns=&quot;http://www.w3.org/2000/svg&quot; width=&quot;800&quot; height=&quot;600&quot;><rect width=&quot;100%&quot; height=&quot;100%&quot; fill=&quot;%23f1f5f9&quot;/><text x=&quot;50%&quot; y=&quot;50%&quot; dominant-baseline=&quot;middle&quot; text-anchor=&quot;middle&quot; fill=&quot;%2394a3b8&quot; font-size=&quot;22&quot; font-family=&quot;Inter&quot;>📷 Gambar tidak tersedia</text></svg>'; ?>"
            alt="<?= e($produk['nama_produk']); ?>"
            loading="lazy"
            onerror="this.src='data:image/svg+xml,<svg xmlns=&quot;http://www.w3.org/2000/svg&quot; width=&quot;800&quot; height=&quot;600&quot;><rect width=&quot;100%&quot; height=&quot;100%&quot; fill=&quot;%23f1f5f9&quot;/><text x=&quot;50%&quot; y=&quot;50%&quot; dominant-baseline=&quot;middle&quot; text-anchor=&quot;middle&quot; fill=&quot;%2394a3b8&quot; font-size=&quot;22&quot; font-family=&quot;Inter&quot;>📷 Gambar tidak tersedia</text></svg>';"
          />
        </div>
        <div class="info">
          <h2><?= e($produk['nama_produk']); ?></h2>
          <div class="price">Rp <?= number_format((float)$produk['harga'], 0, ',', '.'); ?></div>
          <div class="desc"><?= nl2br(e($produk['deskripsi'] ?? '')); ?></div>

          <div class="cta-row">
            <?php
              $waText = "Halo, saya tertarik dengan produk: ".$produk['nama_produk']." (Rp ".number_format((float)$produk['harga'],0,',','.').")";
              $waUrl  = "https://wa.me/6283821470781?text=".rawurlencode($waText);
            ?>
            <a class="btn wa" href="<?= $waUrl; ?>" target="_blank" rel="noopener">
              💬 Pesan via WhatsApp
            </a>
            <a class="btn ghost" href="index.php<?= !empty($produk['kategori_id']) ? '?kategori_id='.(int)$produk['kategori_id'].'#menu':''; ?>">
              ⬅️ Kembali ke Menu
            </a>
          </div>
        </div>
      </section>

      <!-- Related Products -->
      <?php if(!empty($related)): ?>
      <section aria-labelledby="terkait-title" style="margin-bottom:40px">
        <div class="section-title">
          <span class="line"></span>
          <h3 id="terkait-title">Produk Terkait</h3>
          <span class="line"></span>
        </div>
        <div class="product-grid">
          <?php foreach($related as $r): ?>
            <?php
              $rId    = (int)$r['id'];
              $rNama  = e($r['nama_produk']);
              $rHarga = 'Rp '.number_format((float)$r['harga'], 0, ',', '.');
              $rImg   = !empty($r['gambar'])
                        ? 'uploads/'.e($r['gambar'])
                        : 'data:image/svg+xml,<svg xmlns=&quot;http://www.w3.org/2000/svg&quot; width=&quot;600&quot; height=&quot;400&quot;><rect width=&quot;100%&quot; height=&quot;100%&quot; fill=&quot;%23f1f5f9&quot;/><text x=&quot;50%&quot; y=&quot;50%&quot; dominant-baseline=&quot;middle&quot; text-anchor=&quot;middle&quot; fill=&quot;%2394a3b8&quot; font-size=&quot;20&quot; font-family=&quot;Inter&quot;>📷 Gambar tidak tersedia</text></svg>';
            ?>
            <a href="detail.php?id=<?= $rId; ?>" class="product-card-link" aria-label="Detail <?= $rNama; ?>">
              <div class="card">
                <div class="media-sm">
                  <img src="<?= $rImg; ?>" alt="<?= $rNama; ?>" loading="lazy"
                       onerror="this.src='data:image/svg+xml,<svg xmlns=&quot;http://www.w3.org/2000/svg&quot; width=&quot;600&quot; height=&quot;400&quot;><rect width=&quot;100%&quot; height=&quot;100%&quot; fill=&quot;%23f1f5f9&quot;/><text x=&quot;50%&quot; y=&quot;50%&quot; dominant-baseline=&quot;middle&quot; text-anchor=&quot;middle&quot; fill=&quot;%2394a3b8&quot; font-size=&quot;20&quot; font-family=&quot;Inter&quot;>📷 Gambar tidak tersedia</text></svg>';">
                </div>
                <div class="body">
                  <h4><?= $rNama; ?></h4>
                  <div class="price"><?= $rHarga; ?></div>
                </div>
              </div>
            </a>
          <?php endforeach; ?>
        </div>
      </section>
      <?php endif; ?>

    <?php endif; ?>
  </main>

  <footer>
    <p>&copy; <?= date('Y'); ?> Kedai Salsabilla. Semua hak dilindungi undang-undang.</p>
  </footer>
</body>
</html>