<?php
// admin/inventory_api.php — versi diperbaiki

if (session_status() === PHP_SESSION_NONE) session_start();
header('Content-Type: application/json; charset=UTF-8');
ini_set('display_errors', 1);
error_reporting(E_ALL);

require_once __DIR__ . '/../includes/db.php';
require_once __DIR__ . '/auth.php';

/* ---------- helpers ---------- */
function ok($arr = []) {
  echo json_encode(['ok' => true] + $arr, JSON_UNESCAPED_UNICODE);
  exit;
}
function fail($msg, $code = 400) {
  http_response_code($code);
  echo json_encode(['ok' => false, 'message' => $msg], JSON_UNESCAPED_UNICODE);
  exit;
}
function i($k, $d = null) { return $_POST[$k] ?? $d; }

/* ---------- guards ---------- */
if ($_SERVER['REQUEST_METHOD'] !== 'POST') fail('Metode tidak diizinkan', 405);
if (empty($_SESSION['csrf_token'])) $_SESSION['csrf_token'] = bin2hex(random_bytes(16));
if (!hash_equals($_SESSION['csrf_token'], (string)i('csrf',''))) fail('Token invalid. Muat ulang.', 403);

$action = (string)i('action','');

/* ===================================================================
 *  LIST PRODUK
 * ===================================================================*/
if ($action === 'list_produk') {
  $page = max(1, (int)i('page', 1));
  $per  = min(100, max(5, (int)i('per_page', 10)));
  $off  = ($page - 1) * $per;
  $q    = trim((string)i('q', ''));

  if ($q !== '') {
    $stmt = $conn->prepare(
      "SELECT SQL_CALC_FOUND_ROWS p.id,p.nama_produk,p.harga,p.stok,p.gambar,k.nama_kategori
       FROM produk p LEFT JOIN kategori k ON p.kategori_id=k.id
       WHERE p.nama_produk LIKE CONCAT('%',?,'%') OR k.nama_kategori LIKE CONCAT('%',?,'%')
       ORDER BY p.id DESC LIMIT ? OFFSET ?"
    );
    $stmt->bind_param('ssii', $q, $q, $per, $off);
  } else {
    $stmt = $conn->prepare(
      "SELECT SQL_CALC_FOUND_ROWS p.id,p.nama_produk,p.harga,p.stok,p.gambar,k.nama_kategori
       FROM produk p LEFT JOIN kategori k ON p.kategori_id=k.id
       ORDER BY p.id DESC LIMIT ? OFFSET ?"
    );
    $stmt->bind_param('ii', $per, $off);
  }

  if (!$stmt) fail('Query gagal disiapkan', 500);
  $stmt->execute();
  $rows = $stmt->get_result()->fetch_all(MYSQLI_ASSOC);
  $total = (int)$conn->query("SELECT FOUND_ROWS() AS t")->fetch_assoc()['t'];

  ok(['data' => $rows, 'total' => $total, 'page' => $page, 'per_page' => $per]);
}

/* ===================================================================
 *  LIST TRANSAKSI
 * ===================================================================*/
if ($action === 'list_transaksi') {
  $page = max(1, (int)i('page',1));
  $per  = min(100, max(5, (int)i('per_page',10)));
  $off  = ($page - 1) * $per;

  $produk_id = (int)i('produk_id', 0);
  $where = "1=1";
  $binds = []; $types = '';

  if ($produk_id > 0) { $where .= " AND t.produk_id = ?"; $types .= 'i'; $binds[] = $produk_id; }

  $sql =
    "SELECT SQL_CALC_FOUND_ROWS t.*, p.nama_produk
     FROM transaksi t JOIN produk p ON p.id = t.produk_id
     WHERE $where
     ORDER BY t.created_at DESC, t.id DESC
     LIMIT ? OFFSET ?";

  $types .= 'ii'; $binds[] = $per; $binds[] = $off;

  $stmt = $conn->prepare($sql);
  if (!$stmt) fail('Query gagal disiapkan', 500);
  $stmt->bind_param($types, ...$binds);
  $stmt->execute();
  $rows = $stmt->get_result()->fetch_all(MYSQLI_ASSOC);
  $total = (int)$conn->query("SELECT FOUND_ROWS() AS t")->fetch_assoc()['t'];

  ok(['data' => $rows, 'total' => $total, 'page' => $page, 'per_page' => $per]);
}

/* ===================================================================
 *  UPDATE STOK LANGSUNG (± delta)
 * ===================================================================*/
if ($action === 'update_stok') {
  $produk_id = (int)i('produk_id',0);
  $delta     = (int)i('delta',0);
  if ($produk_id <= 0 || $delta === 0) fail('Input tidak valid');

  $conn->begin_transaction();
  try {
    $st = $conn->prepare("SELECT stok FROM produk WHERE id=? FOR UPDATE");
    if (!$st) throw new Exception('Prepare gagal');
    $st->bind_param('i', $produk_id);
    $st->execute();
    $r = $st->get_result()->fetch_assoc();
    if (!$r) { $conn->rollback(); fail('Produk tidak ditemukan',404); }

    $stok_new = (int)$r['stok'] + $delta;
    if ($stok_new < 0) { $conn->rollback(); fail('Stok tidak boleh negatif'); }

    $up = $conn->prepare("UPDATE produk SET stok=? WHERE id=?");
    if (!$up) throw new Exception('Prepare gagal');
    $up->bind_param('ii', $stok_new, $produk_id);
    $up->execute();

    $conn->commit();
    ok(['message' => 'Stok diperbarui', 'stok_baru' => $stok_new]);
  } catch (Throwable $e) {
    $conn->rollback();
    fail('Error: '.$e->getMessage(), 500);
  }
}

/* ===================================================================
 *  ADD TRANSAKSI (mengubah stok)
 * ===================================================================*/
if ($action === 'add_transaksi') {
  $produk_id = (int)i('produk_id',0);
  $tipe      = (string)i('tipe','');
  $qty       = max(1,(int)i('qty',0));
  $customer  = trim((string)i('customer',''));
  $catatan   = trim((string)i('catatan',''));

  if (!in_array($tipe, ['masuk','keluar'], true) || $produk_id <= 0 || $qty <= 0) {
    fail('Input tidak valid');
  }

  $conn->begin_transaction();
  try {
    // kunci stok & ambil harga saat ini
    $st = $conn->prepare("SELECT stok, harga FROM produk WHERE id=? FOR UPDATE");
    if (!$st) throw new Exception('Prepare gagal');
    $st->bind_param('i', $produk_id);
    $st->execute();
    $r = $st->get_result()->fetch_assoc();
    if (!$r) { $conn->rollback(); fail('Produk tidak ada',404); }

    $stok  = (int)$r['stok'];
    $harga = (int)$r['harga'];

    $stok_new = ($tipe === 'keluar') ? ($stok - $qty) : ($stok + $qty);
    if ($stok_new < 0) { $conn->rollback(); fail('Stok tidak cukup'); }

    $up = $conn->prepare("UPDATE produk SET stok=? WHERE id=?");
    if (!$up) throw new Exception('Prepare gagal');
    $up->bind_param('ii', $stok_new, $produk_id);
    $up->execute();

    $total = ($tipe === 'keluar') ? ($qty * $harga) : 0;

    $ins = $conn->prepare(
      "INSERT INTO transaksi (produk_id, tipe, qty, harga_saat_transaksi, total, customer, catatan)
       VALUES (?,?,?,?,?,?,?)"
    );
    if (!$ins) throw new Exception('Prepare gagal');
    $ins->bind_param('isiisss', $produk_id, $tipe, $qty, $harga, $total, $customer, $catatan);
    $ins->execute();

    $conn->commit();
    ok(['message'=>'Transaksi dicatat','stok_baru'=>$stok_new]);
  } catch (Throwable $e) {
    $conn->rollback();
    fail('Error: '.$e->getMessage(), 500);
  }
}

/* ===================================================================
 *  UPDATE TRANSAKSI (HANYA customer & catatan)
 *  -> Tidak mengubah stok/qty/tipe demi konsistensi histori
 * ===================================================================*/
if ($action === 'update_transaksi') {
  $trx_id   = (int)i('id',0);
  $customer = trim((string)i('customer',''));
  $catatan  = trim((string)i('catatan',''));

  if ($trx_id <= 0) fail('Input tidak valid');

  $upd = $conn->prepare("UPDATE transaksi SET customer = ?, catatan = ? WHERE id = ?");
  if (!$upd) fail('Query gagal disiapkan', 500);
  $upd->bind_param('ssi', $customer, $catatan, $trx_id);
  if (!$upd->execute()) fail('Gagal update: '.$upd->error, 500);

  ok(['message' => 'Transaksi diperbarui']);
}

/* ===================================================================
 *  DELETE TRANSAKSI
 *  (Tidak melakukan rekalkulasi stok—histori dibiarkan konsekuen)
 * ===================================================================*/
if ($action === 'delete_transaksi') {
  $id = (int)i('id',0);
  if ($id <= 0) fail('ID tidak valid');

  $d = $conn->prepare("DELETE FROM transaksi WHERE id=?");
  if (!$d) fail('Query gagal disiapkan', 500);
  $d->bind_param('i', $id);
  if (!$d->execute()) fail('Gagal menghapus: '.$conn->error, 500);

  ok(['message' => 'Transaksi dihapus']);
}

/* fallback */
fail('Aksi tidak dikenali', 400);
